<?php 
namespace App\Controllers;

use Tiyyick\Router\Controller;
use Tiyyick\Http\Request;
use Tiyyick\Files\CharAvatar;

use App\Models\Staff;
use App\Models\StaffDocument;
use App\Models\StaffSocial;
use App\Models\Department;
use App\Models\Designation;
use App\Models\Salary;
use App\Models\Social;
use App\Models\Role;
use App\Models\Account;
use App\Forms\LoginForm;
use App\Models\SalaryDetail;
use App\Models\User;
use App\Models\Leave;
use App\Models\LeaveType;
use App\Models\PaymentMode;
use App\Models\Payslip;
use App\Models\StaffAttendance;

/**
 * 
 */
class StaffController extends Controller
{
	public function index()
	{
		$allStaff = Staff::query()->all();

		return render('admin/staff/all-staff', ['all_staff' => $allStaff]);
	}

	public function add()
	{
		$data['designations'] = Designation::query()->all();
		$data['departments'] = Department::query()->all();
		$data['salarys'] = Salary::query()->all();
		$data['roles'] = Role::query()->all();

		return render('admin/staff/add-staff', $data);
	}

	public function create(Request $request)
	{
		$form = validate($request, [
            "firstname" => ["required" => true, "maxlen" => 10],
            "lastname" => ["required" => true, "maxlen" => 10],
            "email" => ["maxlen" => 100, "unique" => Staff::class],
            "phone" => ["unique" => Staff::class],
            "username" => ["required" => true, "unique" => Staff::class],
            "password" => ["required" => true],
            "confirm_password" => ["matches" => "password"],
            "gender" => ["required" => true,],
            "role" => ["required" => true],
            "department" => ["required" => true,],
            "designation" => ["required" => true],
            "staff_num" => ["required" => true,],
            'photo' => ['maxsize' => 2048, 'type' => 'image', 'mimes' => ['jpeg', 'jpg', 'png'],],
        ]);

		if ($form->isValid()) {
			$staff = new Staff($request->body()); //filelog(mt_rand(1000, 9999));

			if ($request->hasFile('photo')) {
				$file = $request->getFile('photo');
				$staff->photo = $file->store('staff', bin2hex(random_bytes(8)) . '.png');
			} else {
				$avatar = new CharAvatar();
				$text = $request->firstname . ' ' . $request->lastname;
				$staff->photo = $avatar->create($text, storage_path('staff', true), bin2hex(random_bytes(8)));
			}

			$newStaff = $staff->save(true); 
			$account = new Account($request->body());
			$account->staff = $newStaff;		
			$account->save();

			$data['ok']  = true; 
			$data['url'] = route('staff.index'); 
			$data['msg'] = 'Staff added successfully';
		} else {
			$data['error'] = $form->getErrors();
		}	

		return json_response($data);	
	}

	public function edit($id)
	{
		$data['staff'] = Staff::query()->get($id);
		$data['designations'] = Designation::query()->all();
		$data['departments'] = Department::query()->all();
		$data['salarys'] = Salary::query()->all();
		$data['roles'] = Role::query()->all();

		return render('admin/staff/edit-staff', $data);
	}

	public function update(Request $request, $id)
	{
		$form = validate($request, [
            "firstname" => ["required" => true, "maxlen" => 100],
            "lastname" => ["required" => true, "maxlen" => 100],
            "email" => ["required" => true, "maxlen" => 100, "unique" => [Staff::class, $id]],
            "phone" => ["required" => true, "unique" => [Staff::class, $id]],
            "password" => ["required" => true],
            "gender" => ["required" => true,],
            'photo' => ['maxsize' => 2048, 'type' => 'image', 'mimes' => ['jpeg', 'jpg', 'png'],],
        ]);

		if ($form->isValid()) {
			$staff = Staff::query()->get($id);
			$staff->update($request->body());

			if ($request->hasFile('photo')) {
				if (file_exists(storage_path("staff/$staff->photo"))) {
	                unlink(storage_path("staff/$staff->photo"));
	            } 
				
				$file = $request->getFile('photo');
				$staff->photo = $file->store('staff', bin2hex(random_bytes(8)) . '.png');
			}

			$staff = $staff->save();
			$account = Account::query()->filter(['staff' => $staff])->one();

			if ($account) {
				$account->update($request->body());
				$account->save();
			}

			if ($staff->id === auth_user()->id && $staff->role === auth_user()->role) {
				auth_refresh($staff);
			}			
			
			$data['ok']  = true; 
			$data['url'] = route('staff.index'); 
			$data['msg'] = 'Staff updated successfully';
		} else {
			$data['error'] = $form->getErrors();
		}	

		return json_response($data);
	}

	public function profile($id)
	{
		$data['staff'] = Staff::query()->get_or404($id);
		$data['socials'] = Social::query()->all();
		$data['bgc'] = [
            'present'  => 'bg-success', 
            'holiday'  => 'bg-info', 
            'late'	   => 'bg-warning', 
            'absent'   => 'bg-danger', 
            'weekend'  => 'bg-primary', 
            'vacation' => 'bg-secondary', 
            'free' 	   => 'bg-ebony-100'
        ]; 
        $data['months'] = ['01'=>'Jan','02'=>'Feb','03'=>'Mar','04'=>'Apr','05'=>'May','06'=>'Jun','07'=>'Jul','08'=>'Aug','09'=>'Sep','10'=>'Oct','11'=>'Nov','12'=>'Dec']; 
		
		return render('admin/staff/profile', $data);
	}

	public function delete(Request $request, $id)
	{
		$form = validate($request);
		if ($form->isValid()) {
			Staff::query()->filter(['id' => $id])->delete();
			$data['ok']  = true; 
			$data['url'] = route('staff.index'); 
			$data['msg'] = 'Staff deleted successfully';
		} else {
			$data['error'] = 'Tampered token or no token';
		}	

		return json_response($data);		
	}

	public function createDocument(Request $request, $id)
	{
		$form = validate($request, [
            "title" => ["required" => true,],
            "type" => ["required" => true,],
            'path' => ["required" => true, 'maxsize' => 2048, 'mimes' => ['jpeg', 'jpg', 'png', 'pdf', 'docx'],],
        ]); 

		if ($form->isValid()) { 
			$staffDoc = new StaffDocument($request->body());
			$staffDoc->staff = $id;

			if ($request->hasFile('path')) {
				$file = $request->getFile('path');
				$staffDoc->path = $file->store('staff-documents', bin2hex(random_bytes(8)) . "." . $file->getExtension());
			} 

			$staffDoc->save();

			return json_response(['ok' => true, 'msg' => 'Staff document added successfully']);
		} else {
			return json_response(['error' => $form->getErrors()]);
		}
	}

	public function updateDocument(Request $request, $id)
	{
		$form = validate($request, [
            "title" => ["required" => true,],
            "type" => ["required" => true,],
            'path' => ['maxsize' => 2048, 'mimes' => ['jpeg', 'jpg', 'png', 'pdf', 'docx'],],
        ]);	

		if ($form->isValid()) {
			$staffDoc = StaffDocument::query()->get_or404($id);
			$staffDoc->update($request->body());

		    if ($request->hasFile('path')) {	
		    	filelog('tester');		    	
		    	if (file_exists(storage_path("staff-documents/$staff->path"))) {
	                unlink(storage_path("staff-documents/$staff->path"));
	            } 
				
				$file = $request->getFile('path');
				$staff->path = $file->store('staff-documents', bin2hex(random_bytes(8)) . "." . $file->getExtension());
		    }
		    
			$staffDoc->save();

			return json_response(['ok' => true, 'msg' => 'Staff document updated successfully']);
		} else {
			return json_response(['error' => $form->getErrors()]);
		}
	}

	public function deleteDocument(Request $request, $id)
	{
		validate($request);
		$staffDoc = StaffDocument::query()->filter(['id' => $id])->one();
		$staff = $staffDoc->staff->id;
		$staffDoc->delete();

		if (file_exists(storage_path("staff-documents/$staffDoc->path"))) {
            unlink(storage_path("staff-documents/$staffDoc->path"));
        } 
				
		return json_response(['ok' => true, 'msg' => 'Staff deleted successfully']);
	}

	public function createHandle(Request $request)
	{
		$form = validate($request, [
            "social" => ["required" => true,],
            "link" => ["required" => true,],
        ]); 

		if ($form->isValid()) { 
			StaffSocial::query()->create($request->body());

			return json_response(['ok' => true, 'msg' => 'Staff social handle added successfully']);
		} else {
			return json_response(['error' => $form->getErrors()]);
		}
	}

	public function updateHandle(Request $request, $id)
	{
		$form = validate($request, [
            "social" => ["required" => true,],
            "link" => ["required" => true,],
        ]);	

		if ($form->isValid()) {
			StaffSocial::query()->filter(['id' => $id])->update($request->body());

			return json_response(['ok' => true, 'msg' => 'Staff social handle updated successfully']);
		} else {
			return json_response(['error' => $form->getErrors()]);
		}
	}

	public function deleteHandle(Request $request, $id)
	{
		validate($request);
		StaffSocial::query()->filter(['id' => $id])->delete();

		return json_response(['ok' => true, 'msg' => 'Staff social handle deleted successfully']);
	}

	/**
	 * STAFF ATTENDANCE
	*/
	public function attendance(Request $request)
	{
		if ($request->hasParam('role') && $request->hasParam('date')) {
			$data['attend_date'] = $request->date;
			$data['getrole'] = $request->role;
			$data['currentrole'] = Role::query()->get($data['getrole']);
			$data['staffs'] = Staff::query()->filter(['department' => $request->role])->all();
			$data['attendances'] = StaffAttendance::query()->filter([
				'date' => $data['attend_date'], 'and',
				'staff:in' => Staff::query()->filter(['department' => $request->role])->project('id')->find(),
			])->all();
		} 
		
		$data['roles'] = Role::query()->all();

		return render('admin/attendance/staff', $data);
	}

	public function saveAttendance(Request $request)
	{
		$valid = validate($request);
		
		if ($valid->isValid()) {
			foreach ($request->input('rows') as $row) {
				$attendance = StaffAttendance::query()->filter([
					'date' => $row['date'], 'and',
					'staff' => $row['staff'],
				])->one();

				if ($attendance) {
					$attendance->update($row);
					$attendance->save();
				} else {
					StaffAttendance::query()->create($row);
				}
			}

		    return json_response([ 'ok' => true, 'msg' => "Staff attendance saved successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function login(Request $request)
	{
		if (is_auth()) {
			return redirect(route('admin.index'));
		}
		
		if ($request->method === 'POST') {
			$loginform = new LoginForm($request);

			if ($loginform->isValid()) {
				$user = Staff::query()->filter(['username' => $request->username, 'or', 'email' => $request->username, 'or', 'phone' => $request->username])->one();

				if ($user && password_verify($request->password, $user->password)) {
					$rm = isset($_POST['remember_me']) ? 1 : 0;
					login_user($user, $rm);
					return redirect(route('admin.index'));
				} else {
					flash('danger', 'Invalid email or password'); 
				}
			} else {
				flash('danger', 'Invalid email or password');
			}
		} else {
			$loginform = new LoginForm();
		}

		return render('admin/staff/login-staff', ['loginform' => $loginform]);
	}

	public function logout()
	{
		logout_user();
		return redirect(route('admin.login'));
	}
	// DEPARTMENT
	public function department()
	{
		$departments = Department::query()->all();

		return render('admin/staff/department', ['departments' => $departments]);
	}

	public function createDepartment(Request $request)
	{
		$valid = validate($request, [
			'name' => ['required' => true, 'unique' => Department::class],
		]);

		if ($valid->isValid()) {
			Department::query()->create($request->body());

		    return json_response(['ok' => true, 'msg' => "Department created successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function updateDepartment(Request $request, $id)
	{
		$valid = validate($request, [
			'name' => ['required' => true, 'unique' => [Department::class, $id]],
		]);

		if ($valid->isValid()) {
			$department = Department::query()->filter(['id' => $id])->one(); 
			$department->name = $request->name;
			$department->remark = $request->remark;
			$department->save();

		    return json_response(['ok' => true, 'msg' => "Department updated successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function deleteDepartment(Request $request, $id)
	{
		$valid = validate($request);

		if ($valid->isValid()) {
			Department::query()->filter(['id' => $id])->delete();

			return json_response(['ok' => true, 'msg' => "Department deleted successfully"]);
		} else {
			return json_response(['token_error' => 'Tampered token or no token']);
		}
	}
	// DESIGNATION
	public function designation()
	{
		$designations = Designation::query()->all();

		return render('admin/staff/designation', ['designations' => $designations]);
	}

	public function createDesignation(Request $request)
	{
		$valid = validate($request, [
			'title' => ['required' => true, 'unique' => Designation::class],
		]);

		if ($valid->isValid()) {
			Designation::query()->create($request->body());

		    return json_response(['ok' => true, 'msg' => "Designation created successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function updateDesignation(Request $request, $id)
	{
		$valid = validate($request, [
			'title' => ['required' => true, 'unique' => [Designation::class, $id]],
		]);

		if ($valid->isValid()) {
			$designation = Designation::query()->filter(['id' => $id])->one(); 
			$designation->title = $request->title;
			$designation->save();

		    return json_response(['ok' => true, 'msg' => "Designation updated successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function deleteDesignation(Request $request, $id)
	{
		$valid = validate($request);

		if ($valid->isValid()) {
			Designation::query()->filter(['id' => $id])->delete();

			return json_response(['ok' => true, 'msg' => "Designation deleted successfully"]);
		} else {
			return json_response(['token_error' => 'Tampered token or no token']);
		}
	}
	// SOCIAL
	public function social()
	{
		$socials = Social::query()->all();

		return render('admin/staff/social', ['socials' => $socials]);
	}

	public function createSocial(Request $request)
	{
		$valid = validate($request, [
			'handle' => ['required' => true, 'unique' => Social::class],
		]);

		if ($valid->isValid()) {
			Social::query()->create($request->body());

		    return json_response(['ok' => true, 'msg' => "Social created successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function updateSocial(Request $request, $id)
	{
		$valid = validate($request, [
			'handle' => ['required' => true, 'unique' => [Social::class, $id]],
		]);

		if ($valid->isValid()) {
			Social::query()->filter(['id' => $id])->update($request->body()); 

		    return json_response(['ok' => true, 'msg' => "Social updated successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function deleteSocial(Request $request, $id)
	{
		$valid = validate($request);

		if ($valid->isValid()) {
			Social::query()->filter(['id' => $id])->delete();

			return json_response(['ok' => true, 'msg' => "Social deleted successfully"]);
		} else {
			return json_response(['token_error' => 'Tampered token or no token']);
		}
	}
	// SALARY TEMPLATE
	public function salary()
	{
		$salarys = Salary::query()->all();

		return render('admin/staff/salary', ['salarys' => $salarys]);
	}

	public function createSalary(Request $request)
	{
		$valid = validate($request, [
			"grade" => ["required" => true],
            "basic" => ["required" => true],
		]);
		
		if ($valid->isValid()) {
			$salary = new Salary($request->body());
			$salary = $salary->save();//filelog();

			if (! empty($request->input('salary_detail'))) {
				foreach ($request->input('salary_detail') as $row) {
					$salary->salary_details->create($row);
				}
			}
			
		    return json_response([
		    	'ok' => true, 
		    	'msg' => "Salary template created successfully"
			]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function editSalary($id)
	{
		$salary = Salary::query()->get($id);
		$deduct = Salary::query()->get($id);
		$totalAllowance = $salary->salary_details->filter(['type' => 'allowance'])->sum('amount') ?? 0;
		$totalDeduction = $deduct->salary_details->filter(['type' => 'deduction'])->sum('amount') ?? 0;
		$netSalary = (((float) $salary->basic + (float) $totalAllowance) - (float) $totalDeduction);

		return render('admin/staff/salary-edit', [
			'salary' => $salary,
			'deduct' => $deduct,
			'total_allowance' => $totalAllowance,
			'total_deduction' => $totalDeduction,
			'net_salary' => $netSalary,
		]);
	}

	public function updateSalary(Request $request, $id)
	{
		$valid = validate($request, [
			"grade" => ["required" => true],
            "basic" => ["required" => true],
		]);
		
		if ($valid->isValid()) {
			$salary = Salary::query()->get($id);
			$salary->update($request->body());
			$salary = $salary->save();

			if (! empty($request->input('idx'))) {
				SalaryDetail::query()->filter([
					'salary' => $salary, 'and',
					'id:not_in' => $request->input('idx')
				])->delete();
			}
			
			if (! empty($request->input('salary_detail'))) {
				foreach ($request->input('salary_detail') as $row) {
					$salaryDetail = SalaryDetail::query()->filter([
						'name' => $row['name'], 'and',
						'amount' => $row['amount'], 'and',
						'type' => $row['type'], 'and',
						'salary' => $salary->id,
					])->one();

					if ($salaryDetail) {
						$salaryDetail->update($row); 
						$salaryDetail->save();
					} else {
						$salary->salary_details->create($row);
					}
				}
			}
			
		    return json_response([
		    	'ok' => true, 
		    	'url' => route('salary.index'), 
		    	'msg' => "Salary template updated successfully",
			]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function deleteSalary(Request $request, $id)
	{
		$form = validate($request);
		if ($form->isValid()) {
			Salary::query()->filter(['id' => $id])->delete();
			return json_response(['ok' => true, 'msg' => "Salary template deleted successfully",]);
		} else {
			return json_response(['token_error' => 'Tampered token or no token']);
		}
	}

	public function leaveType(Request $request)
	{
		if ($request->hasParam('_id')) {
			return json_response(['row' => LeaveType::query()->get($request->_id)]);
		}

		$leave_types = LeaveType::query()->order_by('id|DESC')->all();

		return render('admin/staff/leave-type', ['leave_types' => $leave_types]);
	}

	public function createLeaveType(Request $request)
	{
		$valid = validate($request, [
			'type' => ['required' => true],
		]);

		if ($valid->isValid()) {
			if ($request->hasParam('_id')) {
				LeaveType::query()->filter(['id' => $request->_id])->update($request->body()); 

		    	return json_response(['ok' => true, 'msg' => "Leave type updated successfully"]);
			}

			LeaveType::query()->create($request->body());

		    return json_response(['ok' => true, 'msg' => "Leave type created successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function updateLeaveType(Request $request, $id)
	{
		$valid = validate($request, [
			'type' => ['required' => true],
		]);

		if ($valid->isValid()) {
			LeaveType::query()->filter(['id' => $id])->update($request->body()); 

	    	return json_response(['ok' => true, 'msg' => "Leave type updated successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function deleteLeaveType(Request $request, $id)
	{
		$valid = validate($request);

		if ($valid->isValid()) {
			LeaveType::query()->filter(['id' => $id])->delete();

			return json_response(['ok' => true, 'msg' => "Leave type deleted successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function leave()
	{
		$data['leaves'] = Leave::query()->order_by('id|DESC')->all();
		$data['roles'] = Role::query()->all();
		$data['leave_types'] = LeaveType::query()->all();

		return render('admin/staff/leave', $data);
	}

	public function leaveApply()
	{
		$data['leaves'] = Leave::query()->filter(['staff' => auth_user()->id])->order_by('id|DESC')->all();
		$data['leave_types'] = LeaveType::query()->all();

		return render('admin/staff/leave-apply', $data);
	}

	public function selectStaff(Request $request)
	{
		$role = Role::query()->get($request->role_id);

		return json_response(prettify($role->employees->all()));
	}

	public function createLeave(Request $request)
	{
		$valid = validate($request, [
			'staff' => ['required' => true],
			'leave_type' => ['required' => true],
			'from' => ['required' => true],
			'to' => ['required' => true],
		]);

		if ($valid->isValid()) {
			$leave = new Leave($request->body());

			if ($request->hasFile('file')) {
				$file = $request->getFile('file');
				$leave->file = $file->store('leave', bin2hex(random_bytes(10)) . '.jpg');
			}

			$leave->save();

		    return json_response(['ok' => true, 'msg' => "Leave request created successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors(), 'old' => $request->body()]);
		}
	}

	public function leaveEdit($id)
	{
		$data['leave'] = Leave::query()->get($id);
		$data['leave_types'] = LeaveType::query()->all();
		$data['roles'] = Role::query()->all();

		return render('admin/staff/leave-edit', $data);
	}

	public function leaveApplyEdit($id)
	{
		$data['leave'] = Leave::query()->get($id);
		$data['leave_types'] = LeaveType::query()->all();

		return render('admin/staff/leave-apply-edit', $data);
	}

	public function updateLeave(Request $request, $id)
	{
		$valid = validate($request, [
			'staff' => ['required' => true],
			'leave_type' => ['required' => true],
			'from' => ['required' => true],
			'to' => ['required' => true],
		]);

		if ($valid->isValid()) {
			$leave = Leave::query()->filter(['id' => $id])->one(); 
			$leave->update($request->body());

			if ($request->hasFile('file')) {
				if (file_exists(storage_path("leave/$leave->file"))) {
	                unlink(storage_path("leave/$leave->file"));
	            } 
				
				$file = $request->getFile('file');
				$leave->file = $file->store('leave',  bin2hex(random_bytes(10)). '.jpg');
			}

			$leave->save();

			if ($request->hasParam('apply')) {
				return json_response(['ok' => true, 'url' => route('leave_apply.index'), 'msg' => "Leave request updated successfully"]);
			} else {
				return json_response(['ok' => true, 'url' => route('leave.index'), 'msg' => "Leave request updated successfully"]);
			}
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function deleteLeave(Request $request, $id)
	{
		$valid = validate($request);

		if ($valid->isValid()) {
			Leave::query()->filter(['id' => $id])->delete();

			return json_response(['ok' => true, 'msg' => "Leave request deleted successfully"]);
		} else {
			return json_response(['token_error' => 'Tampered token or no token']);
		}
	}

	public function salaryPayroll(Request $request)
	{
		if ($request->hasParam('role') && $request->hasParam('month')) {
			$data['getrole'] = $request->role;
			$data['getmonth'] = $request->month;
			$data['currentrole'] = Role::query()->filter(['id' => $data['getrole']])->one();
			$data['staffs'] = $data['currentrole']->employees->count() > 0 ? $data['currentrole']->employees->all() : [];
		} 

		$data['roles'] = Role::query()->all();

		return render('admin/staff/salary-payroll', $data);
	}

	public function salaryAllocation(Request $request)
	{
		if ($request->hasParam('role') && $request->hasParam('salary')) {
			$data['getrole'] = $request->role;
			$data['getsalary'] = $request->salary;
			$data['currentsalary'] = Salary::query()->get($data['getsalary']);
			$data['currentrole'] = Role::query()->get($data['getrole']);
			$data['staffs'] = Staff::query()->filter(['role' => $data['getrole']])->all();
		} 

		$data['salarys'] = Salary::query()->all();
		$data['roles'] = Role::query()->all();

		return render('admin/staff/salary-allocation', $data);
	}

	public function saveSalaryAllocation(Request $request)
	{
		$valid = validate($request);
		
		if ($valid->isValid()) {
			foreach ($request->input('rows') as $emplyee) {
				$staff = Staff::query()->filter(['id' => $emplyee,])->one();

				if ($staff) {
					$staff->salary = $request->_id ;
					$staff->save();
				} 
			}

		    return json_response(['ok' => true, 'url' => route('salary_allocation.index'), 'msg' => "Staff allocated salary successfully"]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

	public function salaryPayment(Request $request, $month, $id)
	{
		if ($request->method === 'POST') {
			$valid = validate($request, [
				'payment_mode' => ['required' => true],
			]); 
			
			if ($valid->isValid()) {
				$payslip = new Payslip($request->body());
				$payslip->bill_num = mt_rand(1000, 9999);
				$payslip->status = 'paid';
				$payslip->save();

			    return json_response([
			    	'ok' => true, 
			    	'url' => route('salary_payslip.index', [$id, $month]), 
			    	'msg' => "Staff salary paid successfully"
				]);
			} else {
				return json_response(['error' => $valid->getErrors()]);
			}
		} 
		
		$data['staff'] = Staff::query()->get_or404($id);
		$data['staff_dud'] = Staff::query()->get($id);
		$data['payment_modes'] = PaymentMode::query()->all();
		$data['month_year'] = $month;

		return render('admin/staff/salary-payment', $data);
	}

	public function salaryPayslip(Request $request, $id, $month)
	{
		$month = explode('-', $month);
		$data['staff'] = Staff::query()->get_or404($id);
		$data['staff_dud'] = Staff::query()->get($id);
		$data['payslip'] = Payslip::query()->filter([
			'staff' => $id, 'and',
			'year' => $month[0], 'and',
			'month' => $month[1]
		])->one();

		return render('admin/staff/salary-payslip', $data);
	}

	public function saveSalaryPayment(Request $request, $month, $id)
	{
		$valid = validate($request, [
			'payment_mode' => ['required' => true],
		]);  
		
		if ($valid->isValid()) {
			$payslip = new Payslip($request->body());
			$payslip->bill_num = mt_rand(1000, 9999);
			$payslip->status = 'paid';
			$payslip->save();

		    return json_response([
		    	'ok' => true, 
		    	'url' => route('salary_payslip.index', [$id, $month]), 
		    	'msg' => "Staff salary paid successfully"
			]);
		} else {
			return json_response(['error' => $valid->getErrors()]);
		}
	}

}
